package furny.swing.admin;

import java.awt.Color;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.BevelBorder;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.text.html.HTMLEditorKit;

import com.jme3.math.Vector3f;

import furny.entities.Furniture;
import furny.entities.Tag;
import furny.furndb.FurnDBManager;
import furny.swing.admin.tags.EditFurnitureTagsPanel;
import furny.swing.admin.viewer.IFurnitureViewer;

/**
 * Label that shows furnitures in a table. This is part of the
 * {@link BrowserPanel#FurnitureCellRenderer}.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class FurnitureLabel extends JLabel implements HyperlinkListener {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(FurnitureLabel.class
      .getName());

  private final ImageIcon icon;
  private final JLabel nameLabel;
  private final JLabel descriptionLabel;
  private final JLabel dateLabel;
  private final JLabel idLabel;
  private final JLabel itemNumberLabel;

  private final JLabel priceLabel;

  private final JLabel dimensionsLabel;

  private final JEditorPane tagPane;

  private final Font normalFont;
  private final Font disabledFont;
  private final Color normalColor = Color.BLACK;
  private final Color disabledColor = Color.LIGHT_GRAY;

  private final NumberFormat floatFormat = new DecimalFormat("0.00");

  private final StringBuilder toolTipBuilder = new StringBuilder();
  private final StringBuilder tagsBuilder = new StringBuilder();

  private final DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

  private final IFurnitureBrowser browser;
  private final IFurnitureViewer viewer;
  private final boolean isEditor;

  private Furniture furniture;

  /**
   * Instantiates a new furniture label.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public FurnitureLabel() {
    this(null, null);
  }

  /**
   * Instantiates a new furniture label.
   * 
   * @param browser
   *          the browser
   * @param viewer
   *          the viewer
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public FurnitureLabel(final IFurnitureBrowser browser,
      final IFurnitureViewer viewer) {
    this.browser = browser;
    this.viewer = viewer;

    isEditor = browser != null;

    setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
    setFocusable(false);

    setLayout(new GridBagLayout());
    GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(10, 10, 2, 10);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;
    constraints.anchor = GridBagConstraints.WEST;
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridwidth = 2;

    icon = new ImageIcon();
    add(new JLabel(icon), constraints);

    constraints.weighty = 0d;
    constraints.insets = new Insets(5, 10, 5, 10);
    constraints.gridy++;
    nameLabel = new JLabel();
    nameLabel.setFont(nameLabel.getFont().deriveFont(Font.BOLD));
    add(nameLabel, constraints);

    constraints.gridy++;
    descriptionLabel = new JLabel();
    add(descriptionLabel, constraints);

    constraints.gridy++;
    itemNumberLabel = new JLabel();
    add(itemNumberLabel, constraints);

    constraints.gridy++;
    constraints.gridwidth = 1;
    add(new JLabel("Price"), constraints);

    constraints.gridx++;
    constraints.fill = GridBagConstraints.NONE;
    constraints.anchor = GridBagConstraints.EAST;

    priceLabel = new JLabel();
    add(priceLabel, constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.anchor = GridBagConstraints.WEST;
    add(new JLabel("Dimensions (w,h,l)"), constraints);

    constraints.gridx++;
    constraints.fill = GridBagConstraints.NONE;
    constraints.anchor = GridBagConstraints.EAST;

    dimensionsLabel = new JLabel();
    add(dimensionsLabel, constraints);

    normalFont = descriptionLabel.getFont();
    disabledFont = normalFont.deriveFont(Font.ITALIC);

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.gridwidth = 2;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.anchor = GridBagConstraints.WEST;
    constraints.insets = new Insets(5, 10, 0, 10);
    idLabel = new JLabel();
    idLabel.setFont(normalFont.deriveFont(10f));
    add(idLabel, constraints);

    constraints.gridy++;
    dateLabel = new JLabel();
    dateLabel.setFont(normalFont.deriveFont(10f));
    add(dateLabel, constraints);

    final JPanel tagPanel = new JPanel();
    tagPanel.setOpaque(false);

    constraints.gridy++;
    constraints.gridx = 0;
    constraints.weightx = 1.0d;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.insets = new Insets(10, 10, 10, 10);

    add(tagPanel, constraints);

    tagPanel.setLayout(new GridBagLayout());
    constraints = new GridBagConstraints();
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.weightx = 0.0d;
    constraints.weighty = 1.0d;
    constraints.fill = GridBagConstraints.BOTH;
    constraints.gridwidth = 1;

    tagPanel.add(Box.createVerticalStrut(50), constraints);

    tagPane = new JEditorPane();
    tagPane.setContentType("text/html");
    tagPane.setEditable(false);
    tagPane.setEditorKit(new HTMLEditorKit());
    tagPane.putClientProperty(JEditorPane.HONOR_DISPLAY_PROPERTIES,
        Boolean.TRUE);
    tagPane.setFont(normalFont);

    constraints.gridx++;
    constraints.weightx = 1.0d;

    tagPanel.add(tagPane, constraints);

    constraints.gridx++;
    constraints.fill = GridBagConstraints.NONE;
    constraints.weighty = 0.0d;
    constraints.weightx = 0.0d;
    constraints.anchor = GridBagConstraints.SOUTH;

    if (isEditor) {

      final JButton addTagButton = new JButton(new ActionAddTag());
      addTagButton.setMargin(new Insets(1, 1, 1, 1));
      tagPanel.add(addTagButton, constraints);

      tagPane.addHyperlinkListener(this);

      nameLabel.addMouseListener(new MouseAdapter() {
        @Override
        public void mouseReleased(final MouseEvent e) {
          if (e.getButton() == MouseEvent.BUTTON1) {
            editName();
          }
        }
      });

      descriptionLabel.addMouseListener(new MouseAdapter() {
        @Override
        public void mouseReleased(final MouseEvent e) {
          if (e.getButton() == MouseEvent.BUTTON1) {
            editDescription();
          }
        }
      });

      itemNumberLabel.addMouseListener(new MouseAdapter() {
        @Override
        public void mouseReleased(final MouseEvent e) {
          if (e.getButton() == MouseEvent.BUTTON1) {
            editItemNumber();
          }
        }
      });

      priceLabel.addMouseListener(new MouseAdapter() {
        @Override
        public void mouseReleased(final MouseEvent e) {
          if (e.getButton() == MouseEvent.BUTTON1) {
            editPrice();
          }
        }
      });

      dimensionsLabel.addMouseListener(new MouseAdapter() {
        @Override
        public void mouseReleased(final MouseEvent e) {
          if (e.getButton() == MouseEvent.BUTTON1) {
            editDimension();
          }
        }
      });

      addMouseListener(new PopUpMouseListener());
    } else {
      tagPanel.add(Box.createHorizontalStrut(15), constraints);
    }
  }

  @Override
  public void hyperlinkUpdate(final HyperlinkEvent e) {
    if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
      browser.searchByTag(e.getDescription());
    }
  }

  /**
   * Sets a furniture.
   * 
   * @param f
   *          the new furniture
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setFurniture(final Furniture f) {
    this.furniture = f;
    if (f == null) {
      nameLabel.setText("");
      descriptionLabel.setText("");
      itemNumberLabel.setText("");
      priceLabel.setText("");
      dimensionsLabel.setText("");
      idLabel.setText("");
      dateLabel.setText("");

      setToolTipText("");
    } else {
      if (f.getImage() != null) {
        icon.setImage(f.getImage());
      }
      nameLabel.setText(f.getMetaData().getName());

      String descr = f.getMetaData().getDescription();
      if (descr == null || descr.isEmpty()) {
        descr = "No Description";
        descriptionLabel.setFont(disabledFont);
        descriptionLabel.setForeground(disabledColor);
      } else {
        descriptionLabel.setFont(normalFont);
        descriptionLabel.setForeground(normalColor);
      }

      descriptionLabel.setText(descr);

      String itemNumber = f.getMetaData().getItemNumber();
      if (itemNumber == null || itemNumber.isEmpty()) {
        itemNumber = "No Item Number";
        itemNumberLabel.setFont(disabledFont);
        itemNumberLabel.setForeground(disabledColor);
      } else {
        itemNumberLabel.setFont(normalFont);
        itemNumberLabel.setForeground(normalColor);
      }

      itemNumberLabel.setText(itemNumber);

      priceLabel.setText(floatFormat.format(f.getMetaData().getPrice()));

      final Vector3f d = f.getMetaData().getDimension();

      dimensionsLabel.setText("" + floatFormat.format(d.getX()) + ", "
          + floatFormat.format(d.getY()) + ", " + floatFormat.format(d.getZ()));

      idLabel.setText("<html><b>ID:</b> " + f.getId() + "</html>");

      dateLabel.setText("<html><b>Changed:</b> " + df.format(f.getDate())
          + "</html>");

      toolTipBuilder.setLength(0);

      tagsBuilder.setLength(0);
      tagsBuilder.append("<html><body>");

      final Iterator<Tag> it = new TreeSet<Tag>(f.getMetaData().getTags())
          .iterator();

      if (it.hasNext()) {
        tagPane.setFont(normalFont);
        tagPane.setForeground(normalColor);
        toolTipBuilder.append("<html><b>Tags</b><table>");

      } else {
        tagsBuilder.append("No Tags");
        tagPane.setFont(disabledFont);
        tagPane.setForeground(disabledColor);
        toolTipBuilder.append("<html><b>No Tags</b><table>");
      }

      while (it.hasNext()) {
        final Tag t = it.next();

        toolTipBuilder.append("<tr><td>");
        toolTipBuilder.append(t.getType());
        toolTipBuilder.append("</td><td>");
        toolTipBuilder.append(t.getName());
        toolTipBuilder.append("</td></tr>");

        tagsBuilder.append("<a href=\"");
        tagsBuilder.append(t.getName());
        tagsBuilder.append("\"/>");
        tagsBuilder.append(t.getName());
        tagsBuilder.append("</a>");

        if (it.hasNext()) {
          tagsBuilder.append(", ");
        }
      }

      toolTipBuilder.append("</table></html>");
      setToolTipText(toolTipBuilder.toString());

      tagsBuilder.append("</body></html>");
      tagPane.setText(tagsBuilder.toString());

    }
  }

  /**
   * Edits the name.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void editName() {
    if (furniture != null) {
      final String s = editText(nameLabel);

      if (s != null && !s.isEmpty()) {
        furniture.getMetaData().setName(s);

        FurnDBManager.getInstance().saveFurniture(furniture, true);
      }
    }
  }

  /**
   * Edits the description.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void editDescription() {
    if (furniture != null) {
      final String s = editText(descriptionLabel);

      if (s != null) {
        furniture.getMetaData().setDescription(s);

        FurnDBManager.getInstance().saveFurniture(furniture, true);
      }
    }
  }

  /**
   * Edits the item number.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void editItemNumber() {
    if (furniture != null) {
      final String s = editText(itemNumberLabel);

      if (s != null) {
        furniture.getMetaData().setItemNumber(s);

        FurnDBManager.getInstance().saveFurniture(furniture, true);
      }
    }
  }

  /**
   * Edits the price.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void editPrice() {
    if (furniture != null) {
      final Double d = editPrice(furniture);

      if (d != null) {
        furniture.getMetaData().setPrice(d);

        FurnDBManager.getInstance().saveFurniture(furniture, true);
      }
    }
  }

  /**
   * Edits the dimension.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void editDimension() {
    if (furniture != null) {
      final Vector3f d = editDimension(furniture);

      if (d != null) {
        furniture.getMetaData().setDimension(d);

        FurnDBManager.getInstance().saveFurniture(furniture, true);
      }
    }
  }

  /**
   * Edits the text of a label.
   * 
   * @param label
   *          the label
   * @return the string
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private String editText(final JLabel label) {
    final String textBefore = label.getText();
    final String textAfter = (String) JOptionPane.showInputDialog(
        FurnitureLabel.this, "Type new text", "Edit text",
        JOptionPane.QUESTION_MESSAGE, null, null, textBefore);
    if (!textBefore.equals(textAfter)) {
      return textAfter;
    }

    return null;
  }

  /**
   * Edits the dimension of a furniture.
   * 
   * @param f
   *          the furniture
   * @return the vector3f
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private Vector3f editDimension(final Furniture f) {
    final Vector3f valueBefore = f.getMetaData().getDimension();
    try {
      final JPanel panel = new JPanel();
      panel.setLayout(new GridBagLayout());
      final GridBagConstraints constraints = new GridBagConstraints();
      constraints.gridx = 0;
      constraints.gridy = 0;
      constraints.weightx = 1.0d;
      constraints.weighty = 1.0d;
      constraints.insets = new Insets(5, 5, 5, 5);
      constraints.fill = GridBagConstraints.BOTH;
      constraints.gridwidth = 6;

      panel.add(new JLabel("Edit the dimensions"), constraints);

      constraints.gridx = 0;
      constraints.gridy++;
      constraints.gridwidth = 1;
      panel.add(new JLabel("Width:"), constraints);

      constraints.gridx++;
      final JSpinner widthSpinner = new JSpinner(new SpinnerNumberModel(
          Float.valueOf(valueBefore.getX()), Float.valueOf(0f),
          Float.valueOf(20f), Float.valueOf(1f)));
      widthSpinner.setEditor(new JSpinner.NumberEditor(widthSpinner, "0.00"));
      panel.add(widthSpinner, constraints);

      constraints.gridx++;
      panel.add(new JLabel("Height:"), constraints);

      constraints.gridx++;
      final JSpinner heightSpinner = new JSpinner(new SpinnerNumberModel(
          Float.valueOf(valueBefore.getY()), Float.valueOf(0f),
          Float.valueOf(20f), Float.valueOf(1f)));
      heightSpinner.setEditor(new JSpinner.NumberEditor(heightSpinner, "0.00"));
      panel.add(heightSpinner, constraints);

      constraints.gridx++;
      panel.add(new JLabel("Length:"), constraints);

      constraints.gridx++;
      final JSpinner lengthSpinner = new JSpinner(new SpinnerNumberModel(
          Float.valueOf(valueBefore.getZ()), Float.valueOf(0f),
          Float.valueOf(20f), Float.valueOf(1f)));
      lengthSpinner.setEditor(new JSpinner.NumberEditor(lengthSpinner, "0.00"));
      panel.add(lengthSpinner, constraints);

      final int n = JOptionPane.showConfirmDialog(FurnitureLabel.this, panel,
          "Edit dimensions", JOptionPane.OK_CANCEL_OPTION,
          JOptionPane.QUESTION_MESSAGE);

      final Vector3f newDim = new Vector3f((Float) widthSpinner.getValue(),
          (Float) heightSpinner.getValue(), (Float) lengthSpinner.getValue());

      if (n == JOptionPane.OK_OPTION && !newDim.equals(valueBefore)) {
        return newDim;
      }
    } catch (final Exception e) {
      LOGGER.log(Level.WARNING, "An error occured", e);
    }

    return null;
  }

  /**
   * Edits the price of a furniture.
   * 
   * @param f
   *          the furniture
   * @return the double
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private Double editPrice(final Furniture f) {
    final Double valueBefore = f.getMetaData().getPrice();
    try {
      final JPanel panel = new JPanel();
      panel.setLayout(new GridBagLayout());
      final GridBagConstraints constraints = new GridBagConstraints();
      constraints.gridx = 0;
      constraints.gridy = 0;
      constraints.weightx = 1.0d;
      constraints.weighty = 1.0d;
      constraints.insets = new Insets(5, 5, 5, 5);
      constraints.fill = GridBagConstraints.BOTH;

      panel.add(new JLabel("Edit the price"), constraints);

      constraints.gridy++;

      final JSpinner spinner = new JSpinner(new SpinnerNumberModel(valueBefore,
          Double.valueOf(0d), Double.valueOf(10000d), Double.valueOf(0.01d)));
      spinner.setEditor(new JSpinner.NumberEditor(spinner, "0.00"));

      panel.add(spinner, constraints);

      final int n = JOptionPane.showConfirmDialog(FurnitureLabel.this, panel,
          "Edit price", JOptionPane.OK_CANCEL_OPTION,
          JOptionPane.QUESTION_MESSAGE);

      final Double valueNew = (Double) spinner.getValue();

      if (n == JOptionPane.OK_OPTION && !valueNew.equals(valueBefore)) {
        return valueNew;
      }
    } catch (final Exception e) {
      LOGGER.log(Level.WARNING, "An error occured", e);
    }

    return null;
  }

  /**
   * Action to edit tags of a furniture.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionAddTag extends AbstractAction {

    /**
     * Instantiates a new action add tag.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionAddTag() {
      super("..");
      putValue(SHORT_DESCRIPTION, "Edit tags for furniture");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      final EditFurnitureTagsPanel tp = new EditFurnitureTagsPanel();

      final List<Tag> existingTags = new ArrayList<Tag>(furniture.getMetaData()
          .getTags());
      tp.addTags(existingTags);

      final int n = JOptionPane.showConfirmDialog(null, tp,
          "Edit tags for furniture", JOptionPane.OK_CANCEL_OPTION,
          JOptionPane.PLAIN_MESSAGE);

      if (n == JOptionPane.OK_OPTION) {
        final List<Tag> tags = tp.getSelectedTags();

        FurnDBManager.getInstance().setFurnitureTags(furniture,
            new TreeSet<Tag>(tags));
      }
    }
  }

  /**
   * Mouse listener to display the popup.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class PopUpMouseListener extends MouseAdapter {
    @Override
    public void mouseReleased(final MouseEvent e) {
      if (e.getButton() == MouseEvent.BUTTON3 && e.getClickCount() == 1) {
        new FurnitureLabelPopupMenu(viewer, furniture).show(
            FurnitureLabel.this, e.getX(), e.getY());
      } else if (e.getButton() == MouseEvent.BUTTON1 && e.getClickCount() == 2) {
        viewer.setFurniture(furniture);
      }

      e.consume();
    }
  }
}
